
/*
 * $Id$
 *
 * Copyright (C) PI 2007-2012. All rights reserved.
 *
 * Version: 2.0.5
 *
 */

/**
 * \file 0000_7001_0302_0001.ddl
 * \brief Identification & Maintenance Functions for Profibus PA V3.02.
 * This file contains all definitions for I&M-Data for PA Profile Version 3.02
 *
 * \section iam_Slot1_Data_import How to import
 * Use 'EVERYTHING' for importing of all available parameters
 *
 * Example:
 * \code
 * IMPORT MANUFACTURER __STANDARD, DEVICE_TYPE __SLOT1_IM_DATA, DEVICE_REVISION 0x0302, DD_REVISION 0x0001
 * {
 *   EVERYTHING;
 * }
 * \endcode
 *
 * \section iam1_IMData_IM0_PROFILE_ID_Displ Special notes for the PROFIL_ID parameter
 * The parameter iam1_IMData_IM0_PROFILE_ID is defined with TYPE UNSIGNED_INTEGER(2) for communication. To display
 * the value in a readable form it is automatically converted with the method iam1_IMData_PostReadIM0_PROFILE_ID.
 * The pretty printed PROFIL_ID parameter is placed in the VARIABLE iam1_IMData_IM0_PROFILE_ID_Displ.
 *
 * \section iam1_IMData_SLOT Special notes for the SLOT parameter
 * The value of the local parameter iam1_IMData_SLOT defines the communication slot for using I&M data.
 * The default value must be set (REDEFINE) to the number of the used slot with I&M data.

 * \section iam_Slot1_Data_def_list This file contains the following definitions
 * \subsection iam_Slot1_Data_def_list_variable List of Variables
 * - iam1_IMData_IM0_Header
 * - iam1_IMData_IM0_MANUFACTURER_ID
 * - iam1_IMData_IM0_ORDER_ID
 * - iam1_IMData_IM0_SERIAL_NUMBER
 * - iam1_IMData_IM0_HARDWARE_REVISION
 * - iam1_IMData_IM0_SOFTWARE_REVISION_CHAR
 * - iam1_IMData_IM0_SOFTWARE_REVISION_FUNC
 * - iam1_IMData_IM0_SOFTWARE_REVISION_BUG_FIX
 * - iam1_IMData_IM0_SOFTWARE_REVISION_INTERN
 * - iam1_IMData_IM0_REVISION_COUNTER
 * - iam1_IMData_IM0_PROFILE_ID
 * - iam1_IMData_IM0_PROFILE_SPECIFIC_TYPE
 * - iam1_IMData_IM0_IM_VERSION_Major
 * - iam1_IMData_IM0_IM_VERSION_Minor
 * - iam1_IMData_IM0_IM_SUPPORTED
 * - iam1_IMData_IM1_Header
 * - iam1_IMData_IM1_TAG_FUNCTION
 * - iam1_IMData_IM1_TAG_LOCATION
 * - iam1_IMData_IM2_Header
 * - iam1_IMData_IM2_INSTALLATION_DATE
 * - iam1_IMData_IM2_Reserved
 * - iam1_IMData_IM3_Header
 * - iam1_IMData_IM3_DESCRIPTOR
 * - iam1_IMData_IM4_Header
 * - iam1_IMData_IM4_SIGNATURE
 * - iam1_IMData_PA_IM0_Header
 * - iam1_IMData_PA_IM0_PA_IM_VERSION_Major
 * - iam1_IMData_PA_IM0_PA_IM_VERSION_Minor
 * - iam1_IMData_PA_IM0_PA_HARDWARE_REVISION
 * - iam1_IMData_PA_IM0_PA_SOFTWARE_REVISION
 * - iam1_IMData_PA_IM0_PA_RESERVED
 * - iam1_IMData_PA_IM0_PA_IM_SUPPORTED
 * - iam1_IMData_SLOT
 * - iam1_IMData_IM0_PROFILE_ID_Displ
 * .
 * \subsection iam_Slot1_Data_def_list_command List of Commands
 * - iam1_IMData_IM0_read
 * - iam1_IMData_IM1_read
 * - iam1_IMData_IM2_read
 * - iam1_IMData_IM3_read
 * - iam1_IMData_IM4_read
 * - iam1_IMData_PA_IM0_read
 * .
 * \subsection iam_Slot1_Data_def_list_menu List of Menus
 * - iam1_IMData_IM0_page
 * - iam1_IMData_IM1_page
 * - iam1_IMData_IM2_page
 * - iam1_IMData_IM3_page
 * - iam1_IMData_IM4_page
 * - iam1_IMData_PA_IM0_page
 * - iam1_IMData_Simple_Dialog
 * - iam1_IMData_Simple_Table
 * - iam1_IMData_IM0_table
 * - iam1_IMData_IM0_SOFTWARE_REVISION_group
 * - iam1_IMData_IM0_IM_VERSION_group
 * - iam1_IMData_IM1_table
 * - iam1_IMData_IM2_table
 * - iam1_IMData_IM3_table
 * - iam1_IMData_IM4_table
 * - iam1_IMData_PA_IM0_table
 * - iam1_IMData_PA_IM0_PA_IM_VERSION_group
 * .
 * \subsection iam_Slot1_Data_def_list_method List of Methods
 * - iam1_IMData_PostReadIM0_PROFILE_ID
 * - iam1_IMData_Method_ReadIMData
 * .
 *
 * \section iam_Slot1_Data_upload_download Upload and Download
 * \subsection iam1_IMData_upload_download_upload Upload
 * To upload the I&M Data you can use the following variables for reading all I&M data over a defined algorithm:
 * \code
 * iam1_IMData_IM0_IM_SUPPORTED,
 * iam1_IMData_PA_IM0_PA_IM_SUPPORTED
 * \endcode
 *
 */

#include "pa_import_0302.h"
#include "pa_macros_0302.h"

/** \cond SKIP_THIS */
MANUFACTURER 0x0000, DEVICE_TYPE 0x7001, DEVICE_REVISION 0x0302, DD_REVISION 0x0001
/** \endcond */


VARIABLE iam1_IMData_IM0_Header
{
    LABEL [PI_iam_IMData_IM0_Header_label];
    CLASS CONTAINED;
    HANDLING READ;
    TYPE ASCII(14)
    {
        DEFAULT_VALUE " ";
    }
}

VARIABLE iam1_IMData_IM0_MANUFACTURER_ID
{
    LABEL [PI_IM0_MANUFACTURER_ID_label];
    HELP [PI_IM0_MANUFACTURER_ID_label];
    CLASS CONTAINED;
    HANDLING READ;
    TYPE ENUMERATED(2)
    {
        PI_ENUMS_IAM_MANUFACTURER
    }
}

VARIABLE iam1_IMData_IM0_ORDER_ID
{
    LABEL [PI_IM0_ORDER_ID_label];
    HELP [PI_IM0_ORDER_ID_label];
    CLASS CONTAINED;
    HANDLING READ;
    TYPE VISIBLE(20);
}

VARIABLE iam1_IMData_IM0_SERIAL_NUMBER
{
    LABEL [PI_IM0_SERIAL_NUMBER_label];
    HELP [PI_IM0_SERIAL_NUMBER_label];
    CLASS CONTAINED;
    HANDLING READ;
    TYPE VISIBLE(16);
}

VARIABLE iam1_IMData_IM0_HARDWARE_REVISION
{
    LABEL [PI_IM0_HARDWARE_REVISION_label];
    HELP [PI_IM0_HARDWARE_REVISION_label];
    CLASS CONTAINED;
    HANDLING READ;
    TYPE UNSIGNED_INTEGER(2)
    {
        DEFAULT_VALUE 0;
    }
}

VARIABLE iam1_IMData_IM0_SOFTWARE_REVISION_CHAR
{
    LABEL [PI_iam_IMData_IM0_SOFTWARE_REVISION_CHAR_label];
    HELP [PI_iam_IMData_IM0_SOFTWARE_REVISION_CHAR_help];
    CLASS CONTAINED;
    HANDLING READ;
    TYPE VISIBLE(1)
    {
        DEFAULT_VALUE " ";
    }
}

VARIABLE iam1_IMData_IM0_SOFTWARE_REVISION_FUNC
{
    LABEL [PI_IM0_SOFTWARE_REVISION_FUNC_label];
    HELP [PI_IM0_SOFTWARE_REVISION_FUNC_label];
    CLASS CONTAINED;
    HANDLING READ;
    TYPE UNSIGNED_INTEGER(1)
    {
        DEFAULT_VALUE 0;
    }
}

VARIABLE iam1_IMData_IM0_SOFTWARE_REVISION_BUG_FIX
{
    LABEL [PI_IM0_SOFTWARE_REVISION_BUG_FIX_label];
    HELP [PI_IM0_SOFTWARE_REVISION_BUG_FIX_label];
    CLASS CONTAINED;
    HANDLING READ;
    TYPE UNSIGNED_INTEGER(1)
    {
        DEFAULT_VALUE 0;
    }
}

VARIABLE iam1_IMData_IM0_SOFTWARE_REVISION_INTERN
{
    LABEL [PI_iam_IMData_IM0_SOFTWARE_REVISION_INTERN_label];
    HELP [PI_iam_IMData_IM0_SOFTWARE_REVISION_INTERN_help];
    CLASS CONTAINED;
    HANDLING READ;
    TYPE UNSIGNED_INTEGER(1)
    {
        DEFAULT_VALUE 0;
    }
}

VARIABLE iam1_IMData_IM0_REVISION_COUNTER
{
    LABEL [PI_IM0_REVISION_COUNTER_label];
    HELP [PI_IM0_REVISION_COUNTER_label];
    CLASS CONTAINED;
    HANDLING READ;
    TYPE UNSIGNED_INTEGER(2)
    {
        DEFAULT_VALUE 0;
    }
}

VARIABLE iam1_IMData_IM0_PROFILE_ID
{
    LABEL [PI_IM0_PROFILE_ID_label_1];
    HELP [PI_IM0_PROFILE_ID_label_1];
    CLASS CONTAINED;
    HANDLING READ;
    TYPE UNSIGNED_INTEGER(2)
    {
        DEFAULT_VALUE 0;
    }
    POST_READ_ACTIONS { iam1_IMData_PostReadIM0_PROFILE_ID }
}

VARIABLE iam1_IMData_IM0_PROFILE_SPECIFIC_TYPE
{
    LABEL [PI_iam_IMData_IM0_PROFILE_SPECIFIC_TYPE_label];
    HELP [PI_iam_IMData_IM0_PROFILE_SPECIFIC_TYPE_help];
    CLASS CONTAINED;
    HANDLING READ;
    TYPE ENUMERATED(2)
    {
        IF ((iam1_IMData_IM0_PROFILE_ID >= 0x9700) && (iam1_IMData_IM0_PROFILE_ID <= 0x97FF))	{ PA_PROFILE_SPECIFIC_TYPE}
    }
}

VARIABLE iam1_IMData_IM0_IM_VERSION_Major
{
    LABEL [PI_IM0_IM_VERSION_Major_label];
    HELP [PI_IM0_IM_VERSION_Major_help];
    CLASS CONTAINED;
    HANDLING READ;
    TYPE UNSIGNED_INTEGER(1)
    {
        DEFAULT_VALUE 0;
    }
}

VARIABLE iam1_IMData_IM0_IM_VERSION_Minor
{
    LABEL [PI_IM0_IM_VERSION_Minor_label];
    HELP [PI_IM0_IM_VERSION_Minor_help];
    CLASS CONTAINED;
    HANDLING READ;
    TYPE UNSIGNED_INTEGER(1)
    {
        DEFAULT_VALUE 0;
    }
}

VARIABLE iam1_IMData_IM0_IM_SUPPORTED
{
    LABEL [PI_IM0_IM_SUPPORTED_label];
    HELP [PI_IM0_IM_SUPPORTED_label];
    CLASS CONTAINED;
    HANDLING READ;
    TYPE UNSIGNED_INTEGER(2);
    PRE_READ_ACTIONS { iam1_IMData_PreReadIM0 }
    POST_READ_ACTIONS { iam1_IMData_PostReadIM0 }
}

VARIABLE iam1_IMData_IM1_Header
{
    LABEL [PI_iam_IMData_IM1_Header_label];
    CLASS CONTAINED;
    HANDLING READ;
    TYPE ASCII(14)
    {
        DEFAULT_VALUE " ";
    }
}

VARIABLE iam1_IMData_IM1_TAG_FUNCTION
{
    LABEL [PI_IM1_TAG_FUNCTION_label];
    HELP [PI_IM1_TAG_FUNCTION_label];
    CLASS CONTAINED;
    HANDLING IF ((iam1_IMData_IM0_PROFILE_ID >= 0x9700) && (iam1_IMData_IM0_PROFILE_ID <= 0x97FF)) { READ; } ELSE { READ & WRITE; }
    TYPE VISIBLE(32);
    VALIDITY IF (iam1_IMData_IM0_IM_SUPPORTED & 0b10) { TRUE;} ELSE {FALSE;}
    PRE_READ_ACTIONS { iam1_IMData_PreReadIM1 }
    PRE_WRITE_ACTIONS { iam1_IMData_PreWriteIM1 }
    POST_WRITE_ACTIONS { iam1_IMData_PostWriteAll_IM }
}

VARIABLE iam1_IMData_IM1_TAG_LOCATION
{
    LABEL [PI_IM1_TAG_LOCATION_label];
    HELP [PI_IM1_TAG_LOCATION_label];
    CLASS CONTAINED;
    HANDLING IF ((iam1_IMData_IM0_PROFILE_ID >= 0x9700) && (iam1_IMData_IM0_PROFILE_ID <= 0x97FF)) { READ; } ELSE { READ & WRITE; }
    TYPE VISIBLE(22);
    VALIDITY IF (iam1_IMData_IM0_IM_SUPPORTED & 0b10) { TRUE;} ELSE {FALSE;}
}

VARIABLE iam1_IMData_IM2_Header
{
    LABEL [PI_iam_IMData_IM2_Header_label];
    CLASS CONTAINED;
    HANDLING READ;
    TYPE ASCII(14)
    {
        DEFAULT_VALUE " ";
    }
}

VARIABLE iam1_IMData_IM2_INSTALLATION_DATE
{
    LABEL [PI_IM2_INSTALLATION_DATE_label];
    HELP [PI_IM2_INSTALLATION_DATE_label];
    CLASS CONTAINED;
    HANDLING IF ((iam1_IMData_IM0_PROFILE_ID >= 0x9700) && (iam1_IMData_IM0_PROFILE_ID <= 0x97FF)) { READ; } ELSE { READ & WRITE; }
    TYPE VISIBLE(16);
    VALIDITY IF (iam1_IMData_IM0_IM_SUPPORTED & 0b100) { TRUE;} ELSE {FALSE;}
    PRE_READ_ACTIONS { iam1_IMData_PreReadIM2 }
    PRE_WRITE_ACTIONS { iam1_IMData_PreWriteIM2 }
    POST_WRITE_ACTIONS { iam1_IMData_PostWriteAll_IM }
}

VARIABLE iam1_IMData_IM2_Reserved
{
    LABEL [PI_iam_IMData_IM2_Reserved_label];
    CLASS CONTAINED;
    HANDLING IF ((iam1_IMData_IM0_PROFILE_ID >= 0x9700) && (iam1_IMData_IM0_PROFILE_ID <= 0x97FF)) { READ; } ELSE { READ & WRITE; }
    TYPE ASCII(38);
    VALIDITY IF (iam1_IMData_IM0_IM_SUPPORTED & 0b100) { TRUE;} ELSE {FALSE;}
}

VARIABLE iam1_IMData_IM3_Header
{
    LABEL [PI_iam_IMData_IM3_Header_label];
    CLASS CONTAINED;
    HANDLING READ;
    TYPE ASCII(14)
    {
        DEFAULT_VALUE " ";
    }
}

VARIABLE iam1_IMData_IM3_DESCRIPTOR
{
    LABEL [PI_IM3_DESCRIPTOR_label];
    HELP [PI_IM3_DESCRIPTOR_label];
    CLASS CONTAINED;
    HANDLING IF ((iam1_IMData_IM0_PROFILE_ID >= 0x9700) && (iam1_IMData_IM0_PROFILE_ID <= 0x97FF)) { READ; } ELSE { READ & WRITE; }
    TYPE VISIBLE(54);
    VALIDITY IF (iam1_IMData_IM0_IM_SUPPORTED & 0b1000) { TRUE;} ELSE {FALSE;}
    PRE_READ_ACTIONS { iam1_IMData_PreReadIM3 }
    PRE_WRITE_ACTIONS { iam1_IMData_PreWriteIM3 }
    POST_WRITE_ACTIONS { iam1_IMData_PostWriteAll_IM }
}

VARIABLE iam1_IMData_IM4_Header
{
    LABEL [PI_iam_IMData_IM4_Header_label];
    CLASS CONTAINED;
    HANDLING READ;
    TYPE ASCII(14)
    {
        DEFAULT_VALUE " ";
    }
}

VARIABLE iam1_IMData_IM4_SIGNATURE
{
    LABEL [PI_iam_IMData_IM4_SIGNATURE_label];
    CLASS CONTAINED;
    HANDLING IF ((iam1_IMData_IM0_PROFILE_ID >= 0x9700) && (iam1_IMData_IM0_PROFILE_ID <= 0x97FF)) { READ; } ELSE { READ & WRITE; }
    TYPE VISIBLE(54);
    VALIDITY IF (iam1_IMData_IM0_IM_SUPPORTED & 0b10000) { TRUE;} ELSE {FALSE;}
    PRE_READ_ACTIONS { iam1_IMData_PreReadIM4 }
    PRE_WRITE_ACTIONS { iam1_IMData_PreWriteIM4 }
    POST_WRITE_ACTIONS { iam1_IMData_PostWriteAll_IM }
}

VARIABLE iam1_IMData_PA_IM0_Header
{
    LABEL [PI_iam_IMData_PA_IM0_Header_label];
    CLASS CONTAINED;
    HANDLING READ;
    TYPE ASCII(14)
    {
        DEFAULT_VALUE " ";
    }
}

VARIABLE iam1_IMData_PA_IM0_PA_IM_VERSION_Major
{
    LABEL [PI_IM0_IM_VERSION_Major_label];
    HELP [PI_IM0_IM_VERSION_Major_help];
    CLASS CONTAINED;
    HANDLING READ;
    TYPE UNSIGNED_INTEGER(1)
    {
        DEFAULT_VALUE 0;
    }
    VALIDITY IF (((iam1_IMData_IM0_IM_SUPPORTED & 0b1) == 1) && (iam1_IMData_IM0_PROFILE_ID >= 0x9700) && (iam1_IMData_IM0_PROFILE_ID <= 0x97FF)){ TRUE;} ELSE {FALSE;}
}

VARIABLE iam1_IMData_PA_IM0_PA_IM_VERSION_Minor
{
    LABEL [PI_IM0_IM_VERSION_Minor_label];
    HELP [PI_IM0_IM_VERSION_Minor_help];
    CLASS CONTAINED;
    HANDLING READ;
    TYPE UNSIGNED_INTEGER(1)
    {
        DEFAULT_VALUE 0;
    }
    VALIDITY IF (((iam1_IMData_IM0_IM_SUPPORTED & 0b1) == 1) && (iam1_IMData_IM0_PROFILE_ID >= 0x9700) && (iam1_IMData_IM0_PROFILE_ID <= 0x97FF)){ TRUE;} ELSE {FALSE;}
}

VARIABLE iam1_IMData_PA_IM0_PA_HARDWARE_REVISION
{
    LABEL [PI_iam_IMData_PA_IM0_PA_HARDWARE_REVISION_label];
    HELP [PI_IM0_HARDWARE_REVISION_label];
    CLASS CONTAINED;
    HANDLING READ;
    TYPE VISIBLE(16);
    VALIDITY IF (((iam1_IMData_IM0_IM_SUPPORTED & 0b1) == 1) && (iam1_IMData_IM0_PROFILE_ID >= 0x9700) && (iam1_IMData_IM0_PROFILE_ID <= 0x97FF)){ TRUE;} ELSE {FALSE;}
}

VARIABLE iam1_IMData_PA_IM0_PA_SOFTWARE_REVISION
{
    LABEL [PI_PA_IM0_PA_SOFTWARE_REVISION_label];
    HELP [PI_PA_IM0_PA_SOFTWARE_REVISION_label];
    CLASS CONTAINED;
    HANDLING READ;
    TYPE VISIBLE(16);
    VALIDITY IF (((iam1_IMData_IM0_IM_SUPPORTED & 0b1) == 1) && (iam1_IMData_IM0_PROFILE_ID >= 0x9700) && (iam1_IMData_IM0_PROFILE_ID <= 0x97FF)){ TRUE;} ELSE {FALSE;}
}

VARIABLE iam1_IMData_PA_IM0_PA_RESERVED
{
    LABEL [PI_iam_IMData_PA_IM0_PA_RESERVED_label];
    CLASS CONTAINED;
    HANDLING READ;
    TYPE ASCII(18);
    VALIDITY IF (((iam1_IMData_IM0_IM_SUPPORTED & 0b1) == 1) && (iam1_IMData_IM0_PROFILE_ID >= 0x9700) && (iam1_IMData_IM0_PROFILE_ID <= 0x97FF)){ TRUE;} ELSE {FALSE;}
}

VARIABLE iam1_IMData_PA_IM0_PA_IM_SUPPORTED
{
    LABEL [PI_IM0_IM_SUPPORTED_label];
    HELP [PI_IM0_IM_SUPPORTED_label];
    CLASS CONTAINED;
    HANDLING READ;
    TYPE UNSIGNED_INTEGER(2);
    VALIDITY IF (((iam1_IMData_IM0_IM_SUPPORTED & 0b1) == 1) && (iam1_IMData_IM0_PROFILE_ID >= 0x9700) && (iam1_IMData_IM0_PROFILE_ID <= 0x97FF)){ TRUE;} ELSE {FALSE;}
    PRE_READ_ACTIONS { iam1_IMData_PreReadPA_IM0 }
}

/* COMMANDs */

COMMAND iam1_IMData_IM0_read
{
    SLOT iam1_IMData_SLOT;
    INDEX 255;
    OPERATION READ;
    RESPONSE_CODES DPV1_PA_rsp_codes;
    TRANSACTION
    {
        REQUEST
        {
        }
        REPLY
        {
            iam1_IMData_IM0_Header,
            iam1_IMData_IM0_MANUFACTURER_ID,
            iam1_IMData_IM0_ORDER_ID,
            iam1_IMData_IM0_SERIAL_NUMBER,
            iam1_IMData_IM0_HARDWARE_REVISION,
            iam1_IMData_IM0_SOFTWARE_REVISION_CHAR,
            iam1_IMData_IM0_SOFTWARE_REVISION_FUNC,
            iam1_IMData_IM0_SOFTWARE_REVISION_BUG_FIX,
            iam1_IMData_IM0_SOFTWARE_REVISION_INTERN
,
            iam1_IMData_IM0_REVISION_COUNTER,
            iam1_IMData_IM0_PROFILE_ID,
            iam1_IMData_IM0_PROFILE_SPECIFIC_TYPE,
            iam1_IMData_IM0_IM_VERSION_Major,
            iam1_IMData_IM0_IM_VERSION_Minor
,
            iam1_IMData_IM0_IM_SUPPORTED
        }
    }
}

COMMAND iam1_IMData_IM1_read
{
    SLOT iam1_IMData_SLOT;
    INDEX 255;
    OPERATION READ;
    RESPONSE_CODES DPV1_PA_rsp_codes;
    TRANSACTION
    {
        REQUEST
        {
        }
        REPLY
        {
            iam1_IMData_IM1_Header,
            iam1_IMData_IM1_TAG_FUNCTION,
            iam1_IMData_IM1_TAG_LOCATION
        }
    }
}

COMMAND iam1_IMData_IM2_read
{
    SLOT iam1_IMData_SLOT;
    INDEX 255;
    OPERATION READ;
    RESPONSE_CODES DPV1_PA_rsp_codes;
    TRANSACTION
    {
        REQUEST
        {
        }
        REPLY
        {
            iam1_IMData_IM2_Header,
            iam1_IMData_IM2_INSTALLATION_DATE,
            iam1_IMData_IM2_Reserved
        }
    }
}

COMMAND iam1_IMData_IM3_read
{
    SLOT iam1_IMData_SLOT;
    INDEX 255;
    OPERATION READ;
    RESPONSE_CODES DPV1_PA_rsp_codes;
    TRANSACTION
    {
        REQUEST
        {
        }
        REPLY
        {
            iam1_IMData_IM3_Header,
            iam1_IMData_IM3_DESCRIPTOR
        }
    }
}

COMMAND iam1_IMData_IM4_read
{
    SLOT iam1_IMData_SLOT;
    INDEX 255;
    OPERATION READ;
    RESPONSE_CODES DPV1_PA_rsp_codes;
    TRANSACTION
    {
        REQUEST
        {
        }
        REPLY
        {
            iam1_IMData_IM4_Header,
            iam1_IMData_IM4_SIGNATURE
        }
    }
}

COMMAND iam1_IMData_PA_IM0_read
{
    SLOT iam1_IMData_SLOT;
    INDEX 255;
    OPERATION READ;
    RESPONSE_CODES DPV1_PA_rsp_codes;
    TRANSACTION
    {
        REQUEST
        {
        }
        REPLY
        {
            iam1_IMData_PA_IM0_Header,
            iam1_IMData_PA_IM0_PA_IM_VERSION_Major,
            iam1_IMData_PA_IM0_PA_IM_VERSION_Minor
,
            iam1_IMData_PA_IM0_PA_HARDWARE_REVISION,
            iam1_IMData_PA_IM0_PA_SOFTWARE_REVISION,
            iam1_IMData_PA_IM0_PA_RESERVED,
            iam1_IMData_PA_IM0_PA_IM_SUPPORTED
        }
    }
}

/* BEGIN post EDD */

// Aman, Bman:
// - VARIABLE iam1_IMData_SLOT

VARIABLE iam1_IMData_SLOT
{
    LABEL		[PI_IaM_SLOT_label];
    HELP		[PI_IaM_SLOT_help];
    CLASS		LOCAL;
    TYPE UNSIGNED_INTEGER(1)
    {
        DEFAULT_VALUE 0;
    }
}

// Aman, Bman:
// - VARIABLE iam1_IMData_IM0_PROFILE_ID_Displ

VARIABLE iam1_IMData_IM0_PROFILE_ID_Displ
{
    LABEL	[PI_IM0_PROFILE_ID_label_1];
    HELP	[PI_IM0_PROFILE_ID_label_1];
    CLASS	LOCAL;
    TYPE	VISIBLE(40)
    {
        DEFAULT_VALUE "";
    }
}

VARIABLE iam1_IMData_IaMProtocolExtendedFuncNum
{
    LABEL		"ExtendedFuncNum";
    HELP		"ExtendedFuncNum";
    CLASS		CONTAINED;
    TYPE		UNSIGNED_INTEGER (2)
    {
        DEFAULT_VALUE   2048;   // 8 and 0
    }
    HANDLING READ & WRITE;
}
VARIABLE iam1_IMData_IaMProtocolDataIndex
{
    LABEL		"Write Data Index";
    CLASS		CONTAINED;
    TYPE		UNSIGNED_INTEGER (2)
  {
    DEFAULT_VALUE   65000;
  }
    HANDLING READ & WRITE;
}

/* simple GUI */

// Aman, Bman:
// - MENU iam1_IMData_Simple_Dialog
#define str_Info_IANDM												"|en|Identification and Maintenance is available only for PDM Version 8.0 or higher"	\
    "|de|Identification and Maintenance ist nur fr PDM Version 8.0 oder hher verfgbar."

MENU iam1_IMData_Simple_Dialog
{
    LABEL		"Identification and Maintenance";
    HELP		[PI_IaM_Simple_Dialog_help];
    STYLE		WINDOW;
    ACCESS	ONLINE;
    ITEMS
    {
#ifdef _PDM_
    #if _PDM_ < 800
    str_Info_IANDM
    #else
    iam1_IMData_IM0_page/*,
        iam1_IMData_IM1_page,
        iam1_IMData_IM2_page,
        iam1_IMData_IM3_page,
        iam1_IMData_IM4_page,
        iam1_IMData_PA_IM0_page*/
    #endif
#endif

    }
}

// Aman, Bman:
// - MENU iam1_IMData_Simple_Table

MENU iam1_IMData_Simple_Table
{
    LABEL		[PI_IaM_Simple_Dialog_label];
    HELP		[PI_IaM_Simple_Dialog_help];
    STYLE		TABLE;
    ACCESS	OFFLINE;
    ITEMS
    {
        iam1_IMData_IM0_table,
        iam1_IMData_IM1_table,
        iam1_IMData_IM2_table,
        iam1_IMData_IM3_table,
        iam1_IMData_IM4_table,
        iam1_IMData_PA_IM0_table
    }
}

MENU iam1_IMData_IM0_page
{
    LABEL	"Identification and Maintenance";
    HELP	[PI_IM0_page_help];
    STYLE	PAGE;
    ITEMS
    {
        iam1_IMData_IM0_MANUFACTURER_ID,
        iam1_IMData_IM0_ORDER_ID,
        iam1_IMData_IM0_SERIAL_NUMBER,
        iam1_IMData_IM1_TAG_FUNCTION,
        iam1_IMData_IM1_TAG_LOCATION,
        iam1_IMData_IM2_INSTALLATION_DATE,
        iam1_IMData_IM3_DESCRIPTOR,
    //iam1_IMData_IM4_SIGNATURE,
        iam1_IMData_PA_IM0_PA_HARDWARE_REVISION,
        iam1_IMData_PA_IM0_PA_SOFTWARE_REVISION,
        /*iam1_IMData_IM0_HARDWARE_REVISION,
        iam1_IMData_IM0_SOFTWARE_REVISION_group,*/
        iam1_IMData_IM0_REVISION_COUNTER,
        iam1_IMData_IM0_PROFILE_ID_Displ (READ_ONLY),
        iam1_IMData_IM0_PROFILE_SPECIFIC_TYPE,
        iam1_IMData_IM0_IM_VERSION_group,
        iam1_IMData_PA_IM0_PA_IM_VERSION_group,
        iam1_IMData_IM0_IM_SUPPORTED (HIDDEN)
    }
}


// Aman, Bman:
// - MENU iam1_IMData_IM0_table

MENU iam1_IMData_IM0_table
{
    LABEL	[PI_IM0_page_label];
    HELP	[PI_IM0_page_help];
    STYLE	TABLE;
    ITEMS
    {
        iam1_IMData_IM0_MANUFACTURER_ID,
        iam1_IMData_IM0_ORDER_ID,
        iam1_IMData_IM0_SERIAL_NUMBER,
        iam1_IMData_IM0_HARDWARE_REVISION,
        iam1_IMData_IM0_SOFTWARE_REVISION_group,
        iam1_IMData_IM0_REVISION_COUNTER,
        iam1_IMData_IM0_PROFILE_ID_Displ (READ_ONLY),
        iam1_IMData_IM0_PROFILE_SPECIFIC_TYPE,
        iam1_IMData_IM0_IM_VERSION_group,
        iam1_IMData_IM0_IM_SUPPORTED (HIDDEN)
    }
}

// Aman, Bman:
// - MENU iam1_IMData_IM0_SOFTWARE_REVISION_group

MENU iam1_IMData_IM0_SOFTWARE_REVISION_group
{
    LABEL		[PI_PA_IM0_PA_SOFTWARE_REVISION_label];
    HELP		[PI_PA_IM0_PA_SOFTWARE_REVISION_label];
    STYLE		GROUP;
    ITEMS
    {
        iam1_IMData_IM0_SOFTWARE_REVISION_CHAR,
        iam1_IMData_IM0_SOFTWARE_REVISION_FUNC,
        iam1_IMData_IM0_SOFTWARE_REVISION_BUG_FIX,
        iam1_IMData_IM0_SOFTWARE_REVISION_INTERN
    }
}

// Aman, Bman:
// - MENU iam1_IMData_IM0_IM_VERSION_group

MENU iam1_IMData_IM0_IM_VERSION_group
{
    LABEL		[PI_IM0_IM_VERSION_label];
    HELP		[PI_IM0_IM_VERSION_help];
    STYLE		GROUP;
    ITEMS
    {
        iam1_IMData_IM0_IM_VERSION_Major,
        iam1_IMData_IM0_IM_VERSION_Minor
    }
}

MENU iam1_IMData_IM1_page
{
    LABEL			[PI_IM1_page_label];
    HELP			[PI_IM1_page_help];
    STYLE			PAGE;
    VALIDITY	IF (iam1_IMData_IM0_IM_SUPPORTED & 2)
                            {TRUE;}
                        ELSE
                            {FALSE;}
    ITEMS
    {
        iam1_IMData_IM1_TAG_FUNCTION,
        iam1_IMData_IM1_TAG_LOCATION
    }
}

// Aman, Bman:
// - MENU iam1_IMData_IM1_table

MENU iam1_IMData_IM1_table
{
    LABEL			[PI_IM1_page_label];
    HELP			[PI_IM1_page_help];
    STYLE			TABLE;
    VALIDITY	IF (iam1_IMData_IM0_IM_SUPPORTED & 2)
                            {TRUE;}
                        ELSE
                            {FALSE;}
    ITEMS
    {
        iam1_IMData_IM1_TAG_FUNCTION,
        iam1_IMData_IM1_TAG_LOCATION
    }
}

MENU iam1_IMData_IM2_page
{
    LABEL			[PI_IM2_page_label];
    HELP			[PI_IM2_page_help];
    STYLE			PAGE;
    VALIDITY	IF (iam1_IMData_IM0_IM_SUPPORTED & 4)
                            {TRUE;}
                        ELSE
                            {FALSE;}
    ITEMS
    {
        iam1_IMData_IM2_INSTALLATION_DATE
    }
}

// Aman, Bman:
// - MENU iam1_IMData_IM2_table

MENU iam1_IMData_IM2_table
{
    LABEL			[PI_IM2_page_label];
    HELP			[PI_IM2_page_help];
    STYLE			TABLE;
    VALIDITY	IF (iam1_IMData_IM0_IM_SUPPORTED & 4)
                            {TRUE;}
                        ELSE
                            {FALSE;}
    ITEMS
    {
        iam1_IMData_IM2_INSTALLATION_DATE
    }
}

MENU iam1_IMData_IM3_page
{
    LABEL			[PI_IM3_page_label];
    HELP			[PI_IM3_page_help];
    STYLE			PAGE;
    VALIDITY	IF (iam1_IMData_IM0_IM_SUPPORTED & 8)
                            {TRUE;}
                        ELSE
                            {FALSE;}
    ITEMS
    {
        iam1_IMData_IM3_DESCRIPTOR
    }
}

// Aman, Bman:
// - MENU iam1_IMData_IM3_table

MENU iam1_IMData_IM3_table
{
    LABEL			[PI_IM3_page_label];
    HELP			[PI_IM3_page_help];
    STYLE			TABLE;
    VALIDITY	IF (iam1_IMData_IM0_IM_SUPPORTED & 8)
                            {TRUE;}
                        ELSE
                            {FALSE;}
    ITEMS
    {
        iam1_IMData_IM3_DESCRIPTOR
    }
}

MENU iam1_IMData_IM4_page
{
    LABEL			[PI_IM4_page_label];
    HELP			[PI_IM4_page_help];
    STYLE			PAGE;
    VALIDITY	IF (iam1_IMData_IM0_IM_SUPPORTED & 16)
                            {TRUE;}
                        ELSE
                            {FALSE;}
    ITEMS
    {
        iam1_IMData_IM4_SIGNATURE
    }
}

// Aman, Bman:
// - MENU iam1_IMData_IM4_table

MENU iam1_IMData_IM4_table
{
    LABEL			[PI_IM4_page_label];
    HELP			[PI_IM4_page_help];
    STYLE			TABLE;
    VALIDITY	IF (iam1_IMData_IM0_IM_SUPPORTED & 16)
                            {TRUE;}
                        ELSE
                            {FALSE;}
    ITEMS
    {
        iam1_IMData_IM4_SIGNATURE
    }
}

MENU iam1_IMData_PA_IM0_page
{
    LABEL			[PI_PA_IM0_page_label];
    HELP			[PI_PA_IM0_page_help];
    STYLE			PAGE;
    VALIDITY	IF (((iam1_IMData_IM0_IM_SUPPORTED & 1) == 1) && (iam1_IMData_IM0_PROFILE_ID >= 0x9700) && (iam1_IMData_IM0_PROFILE_ID <= 0x97FF))
                            {TRUE;}
                        ELSE
                            {FALSE;}

    ITEMS
    {
        iam1_IMData_PA_IM0_PA_IM_VERSION_group,
        iam1_IMData_PA_IM0_PA_HARDWARE_REVISION,
        iam1_IMData_PA_IM0_PA_SOFTWARE_REVISION,
        iam1_IMData_PA_IM0_PA_IM_SUPPORTED (HIDDEN)
  }
}

// Aman, Bman:
// - MENU iam1_IMData_PA_IM0_table

MENU iam1_IMData_PA_IM0_table
{
    LABEL			[PI_PA_IM0_page_label];
    HELP			[PI_PA_IM0_page_help];
    STYLE			TABLE;
    VALIDITY	IF (((iam1_IMData_IM0_IM_SUPPORTED & 1) == 1) && (iam1_IMData_IM0_PROFILE_ID >= 0x9700) && (iam1_IMData_IM0_PROFILE_ID <= 0x97FF))
                            {TRUE;}
                        ELSE
                            {FALSE;}

    ITEMS
    {
        iam1_IMData_PA_IM0_PA_IM_VERSION_group,
        iam1_IMData_PA_IM0_PA_HARDWARE_REVISION,
        iam1_IMData_PA_IM0_PA_SOFTWARE_REVISION,
        iam1_IMData_PA_IM0_PA_IM_SUPPORTED (HIDDEN)
  }
}


// Aman, Bman:
// - MENU iam1_IMData_PA_IM0_PA_IM_VERSION_group

MENU iam1_IMData_PA_IM0_PA_IM_VERSION_group
{
    LABEL			[PI_PA_IM0_PA_IM_VERSION_label];
    HELP			[PI_PA_IM0_PA_IM_VERSION_label];
    STYLE			GROUP;
    ITEMS
    {
        iam1_IMData_PA_IM0_PA_IM_VERSION_Major,
        iam1_IMData_PA_IM0_PA_IM_VERSION_Minor
    }
}
/****************************** Common ***************************************************************/
// for set the IM-Index before read data, IaMProtocolDataIndex must be set
COMMAND iam1_IMData_Write_IaMIndex
{
    SLOT        iam1_IMData_SLOT;
    INDEX       255;
    OPERATION   WRITE;
    TRANSACTION
    {
        REQUEST
        {
            iam1_IMData_IaMProtocolExtendedFuncNum,
            iam1_IMData_IaMProtocolDataIndex
        }
        REPLY
        {
        }
    }
}
COMMAND iam1_IMData_Read_IaMIndex
{
    SLOT        iam1_IMData_SLOT;
    INDEX       255;
    OPERATION   READ;
    TRANSACTION
    {
        REQUEST
        {
        }
        REPLY
        {
            iam1_IMData_IaMProtocolExtendedFuncNum,
            iam1_IMData_IaMProtocolDataIndex
        }
    }
}


METHOD iam1_IMData_PreReadIM0
{
    LABEL		"iam1_IMData_PreReadIM0";
    HELP		"Set the IM-Index to 65000 and write this to the device for read IM_Index 0 ";
    CLASS		INPUT;
    ACCESS  ONLINE;
    DEFINITION
    {
        iassign(iam1_IMData_IaMProtocolDataIndex, 65000);
        WriteCommand(iam1_IMData_Write_IaMIndex);
    }
}


METHOD iam1_IMData_PostReadIM0
{
    LABEL		"iam1_IMData_PostReadIM0";
    HELP		"Read all supported IM_Index deppend on IM_SUPPORTED";
    CLASS		INPUT;
    ACCESS	ONLINE;
    DEFINITION
    {
        if ((iam1_IMData_IM0_IM_SUPPORTED & 2) == 2)
        {
            ReadCommand(iam1_IMData_IM1_read);
        }
        if ((iam1_IMData_IM0_IM_SUPPORTED & 4) == 4)
        {
            ReadCommand(iam1_IMData_IM2_read);
        }
        if ((iam1_IMData_IM0_IM_SUPPORTED & 8) == 8)
        {
            ReadCommand(iam1_IMData_IM3_read);
        }
        if ((iam1_IMData_IM0_IM_SUPPORTED & 16) == 16)
        {
            ReadCommand(iam1_IMData_IM4_read);
        }
        if (((iam1_IMData_IM0_IM_SUPPORTED & 1) == 1) && (iam1_IMData_IM0_PROFILE_ID >= 0x9700) && (iam1_IMData_IM0_PROFILE_ID <= 0x97FF))
        {
            //PUT_MESSAGE("PA-Device found");
            ReadCommand(iam1_IMData_PA_IM0_read);
        }
        // add here other Profile IM0
        save_values();
    }
}

// Aman, Bman:
// - METHOD iam1_IMData_PostReadIM0_PROFILE_ID

METHOD iam1_IMData_PostReadIM0_PROFILE_ID
{
    LABEL		"PROFILE_ID display string";
    HELP		"Calculate PROFILE_ID display string";
    CLASS		INPUT;
    ACCESS	ONLINE;
    DEFINITION
    {
        iam1_IMData_IM0_PROFILE_ID_Displ = "Reserved";
        if (iam1_IMData_IM0_PROFILE_ID == 0x0000)
            iam1_IMData_IM0_PROFILE_ID_Displ = "Non-Profile Device";
        if ((iam1_IMData_IM0_PROFILE_ID >= 0x3A00) && (iam1_IMData_IM0_PROFILE_ID <= 0x3AFF))
            iam1_IMData_IM0_PROFILE_ID_Displ = "PROFIdrive";
        if ((iam1_IMData_IM0_PROFILE_ID >= 0x3B00) && (iam1_IMData_IM0_PROFILE_ID <= 0x3BFF))
            iam1_IMData_IM0_PROFILE_ID_Displ = "Robots and Numeric Controls";
        if ((iam1_IMData_IM0_PROFILE_ID >= 0x3C00) && (iam1_IMData_IM0_PROFILE_ID <= 0x3CFF))
            iam1_IMData_IM0_PROFILE_ID_Displ = "Display & Panel devices";
        if ((iam1_IMData_IM0_PROFILE_ID >= 0x3D00) && (iam1_IMData_IM0_PROFILE_ID <= 0x3DFF))
            iam1_IMData_IM0_PROFILE_ID_Displ = "Encoder";
        if ((iam1_IMData_IM0_PROFILE_ID >= 0x3E00) && (iam1_IMData_IM0_PROFILE_ID <= 0x3EFF))
            iam1_IMData_IM0_PROFILE_ID_Displ = "Optical Safety Devices";
        if ((iam1_IMData_IM0_PROFILE_ID >= 0x3F00) && (iam1_IMData_IM0_PROFILE_ID <= 0x3FFF))
            iam1_IMData_IM0_PROFILE_ID_Displ = "Fluid Power / Hydraulics";
        if ((iam1_IMData_IM0_PROFILE_ID >= 0x4600) && (iam1_IMData_IM0_PROFILE_ID <= 0x46FF))
            iam1_IMData_IM0_PROFILE_ID_Displ = "Fieldbus Integration (API)";
        if ((iam1_IMData_IM0_PROFILE_ID >= 0x4900) && (iam1_IMData_IM0_PROFILE_ID <= 0x49FF))
            iam1_IMData_IM0_PROFILE_ID_Displ = "Water Treatment";
        if ((iam1_IMData_IM0_PROFILE_ID >= 0x4E00) && (iam1_IMData_IM0_PROFILE_ID <= 0x4EFF))
            iam1_IMData_IM0_PROFILE_ID_Displ = "IO-Link Master";
        if ((iam1_IMData_IM0_PROFILE_ID >= 0x4B00) && (iam1_IMData_IM0_PROFILE_ID <= 0x4BFF))
            iam1_IMData_IM0_PROFILE_ID_Displ = "Train Applications";
        if ((iam1_IMData_IM0_PROFILE_ID >= 0x4C00) && (iam1_IMData_IM0_PROFILE_ID <= 0x4CFF))
            iam1_IMData_IM0_PROFILE_ID_Displ = "Remote I/O for Factory Automation";
        if ((iam1_IMData_IM0_PROFILE_ID >= 0x5A00) && (iam1_IMData_IM0_PROFILE_ID <= 0x5AFF))
            iam1_IMData_IM0_PROFILE_ID_Displ = "SEMI-Devices";
        if ((iam1_IMData_IM0_PROFILE_ID >= 0x5B00) && (iam1_IMData_IM0_PROFILE_ID <= 0x5BFF))
            iam1_IMData_IM0_PROFILE_ID_Displ = "Identification Systems";
        if ((iam1_IMData_IM0_PROFILE_ID >= 0x5C00) && (iam1_IMData_IM0_PROFILE_ID <= 0x5CFF))
            iam1_IMData_IM0_PROFILE_ID_Displ = "Weighing & Dosage Systems";
        if ((iam1_IMData_IM0_PROFILE_ID >= 0x5D00) && (iam1_IMData_IM0_PROFILE_ID <= 0x5DFF))
            iam1_IMData_IM0_PROFILE_ID_Displ = "Intelligent Pumps";
        if ((iam1_IMData_IM0_PROFILE_ID >= 0x5E00) && (iam1_IMData_IM0_PROFILE_ID <= 0x5EFF))
            iam1_IMData_IM0_PROFILE_ID_Displ = "Low Voltage Switchgear";
        if ((iam1_IMData_IM0_PROFILE_ID >= 0x5F00) && (iam1_IMData_IM0_PROFILE_ID <= 0x5FFF))
            iam1_IMData_IM0_PROFILE_ID_Displ = "Remote I/O for Process Automation";
        if ((iam1_IMData_IM0_PROFILE_ID >= 0x6200) && (iam1_IMData_IM0_PROFILE_ID <= 0x62FF))
            iam1_IMData_IM0_PROFILE_ID_Displ = "Laboratory Devices";
        if ((iam1_IMData_IM0_PROFILE_ID >= 0x9700) && (iam1_IMData_IM0_PROFILE_ID <= 0x97FF))
            iam1_IMData_IM0_PROFILE_ID_Displ = "PA Devices (for Process Control)";
        if ((iam1_IMData_IM0_PROFILE_ID >= 0xF600) && (iam1_IMData_IM0_PROFILE_ID <= 0xF6FF))
            iam1_IMData_IM0_PROFILE_ID_Displ = "Generic device";
        save_values();
    }
}

METHOD iam1_IMData_PreReadIM1
{
    LABEL		"iam1_IMData_PreReadIM1";
    HELP		"Set the IM-Index to 65001 and write this to the device for read IM_Index 1";
    CLASS		INPUT;
    ACCESS	ONLINE;
    DEFINITION
    {
        iassign(iam1_IMData_IaMProtocolDataIndex, 65001);
        WriteCommand(iam1_IMData_Write_IaMIndex);
    }
}
METHOD iam1_IMData_PreWriteIM1
{
    LABEL		"iam1_IMData_PreWriteIM1";
    HELP		"Set the local parameter to 65001 for the following command to write IM_Index 1 to the device";
    CLASS		INPUT;
    ACCESS  ONLINE;
    DEFINITION
    {
        iassign(iam1_IMData_IaMProtocolDataIndex, 65001);
    }
}
METHOD iam1_IMData_PreReadIM2
{
    LABEL		"iam1_IMData_PreReadIM2";
    HELP		"Set the IM-Index to 65002 and write this to the device for read IM_Index 2";
    CLASS		INPUT;
    ACCESS	ONLINE;
    DEFINITION
    {
        iassign(iam1_IMData_IaMProtocolDataIndex, 65002);
        WriteCommand(iam1_IMData_Write_IaMIndex);
    }
}
METHOD iam1_IMData_PreWriteIM2
{
    LABEL		"iam1_IMData_PreWriteIM2";
    HELP		"Set the local parameter to 65002 for the following command to write IM_Index 2 to the device";
    CLASS		INPUT;
    ACCESS  ONLINE;
    DEFINITION
    {
        iassign(iam1_IMData_IaMProtocolDataIndex,65002);
    }
}
METHOD iam1_IMData_PreReadIM3
{
    LABEL		"iam1_IMData_PreReadIM3";
    HELP		"Set the IM-Index to 65003 and write this to the device for read IM_Index 3";
    CLASS		INPUT;
    ACCESS	ONLINE;
    DEFINITION
    {
        iassign(iam1_IMData_IaMProtocolDataIndex,65003);
        WriteCommand(iam1_IMData_Write_IaMIndex);
    }
}
METHOD iam1_IMData_PreWriteIM3
{
    LABEL		"iam1_IMData_PreWriteIM3";
    HELP		"Set the local parameter to 65003 for the following command to write IM_Index 3 to the device";
    CLASS		INPUT;
    ACCESS  ONLINE;
    DEFINITION
    {
        iassign(iam1_IMData_IaMProtocolDataIndex,65003);
    }
}
METHOD iam1_IMData_PreReadIM4
{
    LABEL		"iam1_IMData_PreReadIM4";
    HELP		"Set the IM-Index to 65004 and write this to the device for read IM_Index 4";
    CLASS		INPUT;
    ACCESS	ONLINE;
    DEFINITION
    {
        iassign(iam1_IMData_IaMProtocolDataIndex,65004);
        WriteCommand(iam1_IMData_Write_IaMIndex);
    }
}
METHOD iam1_IMData_PreWriteIM4
{
    LABEL		"iam1_IMData_PreWriteIM4";
    HELP		"Set the local parameter to 65004 for the following command to write IM_Index 4 to the device";
    CLASS		INPUT;
    ACCESS  ONLINE;
    DEFINITION
    {
        iassign(iam1_IMData_IaMProtocolDataIndex,65004);
    }
}
/************************ PA-Devices ***********************************/

METHOD iam1_IMData_PreReadPA_IM0
{
    LABEL		"iam1_IMData_PreReadPA_IM0";
    HELP		"Set the IM-Index to 65016 and write this to the device for read PA IM_Index 0";
    CLASS		INPUT;
    ACCESS	ONLINE;
    DEFINITION
    {
        iassign(iam1_IMData_IaMProtocolDataIndex, 65016);
        WriteCommand(iam1_IMData_Write_IaMIndex);
    }
}

METHOD iam1_IMData_PostWriteAll_IM
{
    LABEL		"iam1_IMData_PostWriteAll_IM";
    HELP		"ReRead the written IM_Index";
    CLASS		INPUT;
    ACCESS  ONLINE;
    DEFINITION
    {
//		ACKNOWLEDGE("PostWriteAll_IM");
        ReadCommand(iam1_IMData_Read_IaMIndex);
    }
}

// Aman, Bman:
// - METHOD iam1_IMData_Method_ReadIMData

METHOD iam1_IMData_Method_ReadIMData
{
    LABEL		"iam1_IMData_ReadIM_Data";
    HELP		"Read IM0 and than all supported IM_Index";
    CLASS		INPUT;
    ACCESS  ONLINE;
    DEFINITION
    {
        ReadCommand(iam1_IMData_IM0_read);
    }
}
/* END post EDD */
